/****************************************************************************
 * drivers/ipcc/ipcc_open.c
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>
#include <nuttx/ipcc.h>

#include <assert.h>
#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <sys/types.h>

#include "ipcc_priv.h"

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: ipcc_open
 *
 * Description:
 *   Opens driver for use by userspace applications.
 *
 * Input Parameters:
 *   filep - pointer to a file structure to open.
 *
 * Returned Value:
 *   OK on successful open, or negated errno on failure.
 *
 * Assumptions/Limitations:
 *
 ****************************************************************************/

int ipcc_open(FAR struct file *filep)
{
  FAR struct ipcc_driver_s *priv;
  int ret;

  /* Get our private data structure */

  priv = filep->f_inode->i_private;

  /* Get exclusive access to the IPCC driver state structure */

  if ((ret = nxmutex_lock(&priv->lock)) < 0)
    {
      return ret;
    }

  /* Increment the count of open references on the driver */

  priv->crefs++;
  DEBUGASSERT(priv->crefs > 0);

  nxmutex_unlock(&priv->lock);
  return OK;
}
