/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.fs.gs.storage;

import com.google.api.gax.paging.Page;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.fs.gs.storage.GSBlobIdentifier;
import org.apache.flink.fs.gs.storage.GSBlobStorage;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GSBlobStorageImpl
implements GSBlobStorage {
    private static final Logger LOGGER = LoggerFactory.getLogger(GSBlobStorageImpl.class);
    private final Storage storage;

    public GSBlobStorageImpl(Storage storage) {
        LOGGER.debug("Creating GSBlobStorageImpl");
        this.storage = (Storage)Preconditions.checkNotNull((Object)storage);
    }

    @Override
    public GSBlobStorage.WriteChannel writeBlob(GSBlobIdentifier blobIdentifier) {
        LOGGER.trace("Creating writable blob for identifier {}", (Object)blobIdentifier);
        Preconditions.checkNotNull((Object)blobIdentifier);
        BlobInfo blobInfo = BlobInfo.newBuilder(blobIdentifier.getBlobId()).build();
        com.google.cloud.WriteChannel writeChannel = this.storage.writer(blobInfo, new Storage.BlobWriteOption[0]);
        return new WriteChannel(blobIdentifier, writeChannel);
    }

    @Override
    public GSBlobStorage.WriteChannel writeBlob(GSBlobIdentifier blobIdentifier, MemorySize chunkSize) {
        LOGGER.trace("Creating writable blob for identifier {} with chunk size {}", (Object)blobIdentifier, (Object)chunkSize);
        Preconditions.checkNotNull((Object)blobIdentifier);
        Preconditions.checkArgument((chunkSize.getBytes() > 0L ? 1 : 0) != 0);
        BlobInfo blobInfo = BlobInfo.newBuilder(blobIdentifier.getBlobId()).build();
        com.google.cloud.WriteChannel writeChannel = this.storage.writer(blobInfo, new Storage.BlobWriteOption[0]);
        writeChannel.setChunkSize((int)chunkSize.getBytes());
        return new WriteChannel(blobIdentifier, writeChannel);
    }

    @Override
    public void createBlob(GSBlobIdentifier blobIdentifier) {
        LOGGER.trace("Creating empty blob {}", (Object)blobIdentifier);
        Preconditions.checkNotNull((Object)blobIdentifier);
        BlobInfo blobInfo = BlobInfo.newBuilder(blobIdentifier.getBlobId()).build();
        this.storage.create(blobInfo, new Storage.BlobTargetOption[0]);
    }

    @Override
    public Optional<GSBlobStorage.BlobMetadata> getMetadata(GSBlobIdentifier blobIdentifier) {
        LOGGER.trace("Getting metadata for blob {}", (Object)blobIdentifier);
        Preconditions.checkNotNull((Object)blobIdentifier);
        Optional<GSBlobStorage.BlobMetadata> metadata = Optional.ofNullable(this.storage.get(blobIdentifier.getBlobId())).map(blob -> new BlobMetadata(blobIdentifier, (Blob)blob));
        if (metadata.isPresent()) {
            LOGGER.trace("Found metadata for blob {}", (Object)blobIdentifier);
        } else {
            LOGGER.trace("Did not find metadata for blob {}", (Object)blobIdentifier);
        }
        return metadata;
    }

    @Override
    public List<GSBlobIdentifier> list(String bucketName, String objectPrefix) {
        LOGGER.trace("Listing blobs in bucket {} with object prefix {}", (Object)bucketName, (Object)objectPrefix);
        Preconditions.checkNotNull((Object)bucketName);
        Preconditions.checkNotNull((Object)objectPrefix);
        Page<Blob> blobs = this.storage.list(bucketName, Storage.BlobListOption.prefix(objectPrefix));
        List<GSBlobIdentifier> blobIdentifiers = StreamSupport.stream(blobs.iterateAll().spliterator(), false).map(BlobInfo::getBlobId).map(GSBlobIdentifier::fromBlobId).collect(Collectors.toList());
        LOGGER.trace("Found blobs in bucket {} with object prefix {}: {}", new Object[]{bucketName, objectPrefix, blobIdentifiers});
        return blobIdentifiers;
    }

    @Override
    public void copy(GSBlobIdentifier sourceBlobIdentifier, GSBlobIdentifier targetBlobIdentifier) {
        LOGGER.trace("Copying blob {} to blob {}", (Object)sourceBlobIdentifier, (Object)targetBlobIdentifier);
        Preconditions.checkNotNull((Object)sourceBlobIdentifier);
        Preconditions.checkNotNull((Object)targetBlobIdentifier);
        this.storage.get(sourceBlobIdentifier.getBlobId()).copyTo(targetBlobIdentifier.getBlobId(), new Blob.BlobSourceOption[0]).getResult();
    }

    @Override
    public void compose(List<GSBlobIdentifier> sourceBlobIdentifiers, GSBlobIdentifier targetBlobIdentifier) {
        LOGGER.trace("Composing blobs {} to blob {}", sourceBlobIdentifiers, (Object)targetBlobIdentifier);
        Preconditions.checkNotNull(sourceBlobIdentifiers);
        Preconditions.checkArgument((sourceBlobIdentifiers.size() > 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((sourceBlobIdentifiers.size() <= 32 ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)targetBlobIdentifier);
        Storage.ComposeRequest.Builder builder = Storage.ComposeRequest.newBuilder();
        BlobInfo targetBlobInfo = BlobInfo.newBuilder(targetBlobIdentifier.getBlobId()).build();
        builder.setTarget(targetBlobInfo);
        for (GSBlobIdentifier blobIdentifier : sourceBlobIdentifiers) {
            builder.addSource(blobIdentifier.objectName);
        }
        Storage.ComposeRequest request = builder.build();
        this.storage.compose(request);
    }

    @Override
    public List<Boolean> delete(Iterable<GSBlobIdentifier> blobIdentifiers) {
        LOGGER.trace("Deleting blobs {}", blobIdentifiers);
        Preconditions.checkNotNull(blobIdentifiers);
        Iterable blobIds = StreamSupport.stream(blobIdentifiers.spliterator(), false).map(GSBlobIdentifier::getBlobId).collect(Collectors.toList());
        return this.storage.delete(blobIds);
    }

    static class WriteChannel
    implements GSBlobStorage.WriteChannel {
        private final GSBlobIdentifier blobIdentifier;
        private final com.google.cloud.WriteChannel writeChannel;

        private WriteChannel(GSBlobIdentifier blobIdentifier, com.google.cloud.WriteChannel writeChannel) {
            this.blobIdentifier = (GSBlobIdentifier)Preconditions.checkNotNull((Object)blobIdentifier);
            this.writeChannel = (com.google.cloud.WriteChannel)Preconditions.checkNotNull((Object)writeChannel);
        }

        @Override
        public int write(byte[] content, int start, int length) throws IOException {
            LOGGER.trace("Writing {} bytes to blob {}", (Object)length, (Object)this.blobIdentifier);
            Preconditions.checkNotNull((Object)content);
            Preconditions.checkArgument((start >= 0 ? 1 : 0) != 0);
            Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0);
            ByteBuffer byteBuffer = ByteBuffer.wrap(content, start, length);
            int written = this.writeChannel.write(byteBuffer);
            LOGGER.trace("Wrote {} bytes to blob {}", (Object)written, (Object)this.blobIdentifier);
            return written;
        }

        @Override
        public void close() throws IOException {
            LOGGER.trace("Closing write channel to blob {}", (Object)this.blobIdentifier);
            this.writeChannel.close();
        }
    }

    static class BlobMetadata
    implements GSBlobStorage.BlobMetadata {
        private final GSBlobIdentifier blobIdentifier;
        private final Blob blob;

        private BlobMetadata(GSBlobIdentifier blobIdentifier, Blob blob) {
            this.blobIdentifier = (GSBlobIdentifier)Preconditions.checkNotNull((Object)blobIdentifier);
            this.blob = (Blob)Preconditions.checkNotNull((Object)blob);
        }

        @Override
        public String getChecksum() {
            LOGGER.trace("Getting checksum for blob {}", (Object)this.blobIdentifier);
            String checksum = this.blob.getCrc32c();
            LOGGER.trace("Found checksum for blob {}: {}", (Object)this.blobIdentifier, (Object)checksum);
            return checksum;
        }
    }
}

