/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.ActiveCompactionsTracker;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.writers.CompactionAwareWriter;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.ISSTableScanner;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.repair.consistent.admin.CleanupSummary;
import org.apache.cassandra.schema.CompactionParams;
import org.apache.cassandra.service.ActiveRepairService;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class PendingRepairManager {
    private static final Logger logger = LoggerFactory.getLogger(PendingRepairManager.class);
    private final ColumnFamilyStore cfs;
    private final CompactionParams params;
    private final boolean isTransient;
    private volatile ImmutableMap<UUID, AbstractCompactionStrategy> strategies = ImmutableMap.of();

    PendingRepairManager(ColumnFamilyStore cfs, CompactionParams params, boolean isTransient) {
        this.cfs = cfs;
        this.params = params;
        this.isTransient = isTransient;
    }

    private ImmutableMap.Builder<UUID, AbstractCompactionStrategy> mapBuilder() {
        return ImmutableMap.builder();
    }

    AbstractCompactionStrategy get(UUID id) {
        return this.strategies.get(id);
    }

    AbstractCompactionStrategy get(SSTableReader sstable) {
        assert (sstable.isPendingRepair());
        return this.get(sstable.getSSTableMetadata().pendingRepair);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    AbstractCompactionStrategy getOrCreate(UUID id) {
        PendingRepairManager.checkPendingID(id);
        assert (id != null);
        AbstractCompactionStrategy strategy = this.get(id);
        if (strategy == null) {
            PendingRepairManager pendingRepairManager = this;
            synchronized (pendingRepairManager) {
                strategy = this.get(id);
                if (strategy == null) {
                    logger.debug("Creating {}.{} compaction strategy for pending repair: {}", new Object[]{this.cfs.metadata.keyspace, this.cfs.metadata.name, id});
                    strategy = this.cfs.createCompactionStrategyInstance(this.params);
                    this.strategies = this.mapBuilder().putAll(this.strategies).put(id, strategy).build();
                }
            }
        }
        return strategy;
    }

    private static void checkPendingID(UUID pendingID) {
        if (pendingID == null) {
            throw new IllegalSSTableArgumentException("sstable is not pending repair");
        }
    }

    AbstractCompactionStrategy getOrCreate(SSTableReader sstable) {
        return this.getOrCreate(sstable.getSSTableMetadata().pendingRepair);
    }

    private synchronized void removeSessionIfEmpty(UUID sessionID) {
        if (!this.strategies.containsKey(sessionID) || !this.strategies.get(sessionID).getSSTables().isEmpty()) {
            return;
        }
        logger.debug("Removing compaction strategy for pending repair {} on  {}.{}", new Object[]{sessionID, this.cfs.metadata.keyspace, this.cfs.metadata.name});
        this.strategies = ImmutableMap.copyOf(Maps.filterKeys(this.strategies, k -> !k.equals(sessionID)));
    }

    synchronized void removeSSTable(SSTableReader sstable) {
        for (Map.Entry entry : this.strategies.entrySet()) {
            ((AbstractCompactionStrategy)entry.getValue()).removeSSTable(sstable);
            this.removeSessionIfEmpty((UUID)entry.getKey());
        }
    }

    void removeSSTables(Iterable<SSTableReader> removed) {
        for (SSTableReader sstable : removed) {
            this.removeSSTable(sstable);
        }
    }

    synchronized void addSSTable(SSTableReader sstable) {
        Preconditions.checkArgument(sstable.isTransient() == this.isTransient);
        this.getOrCreate(sstable).addSSTable(sstable);
    }

    void addSSTables(Iterable<SSTableReader> added) {
        for (SSTableReader sstable : added) {
            this.addSSTable(sstable);
        }
    }

    synchronized void replaceSSTables(Set<SSTableReader> removed, Set<SSTableReader> added) {
        UUID sessionID;
        if (removed.isEmpty() && added.isEmpty()) {
            return;
        }
        HashMap groups = new HashMap();
        for (SSTableReader sSTableReader : removed) {
            sessionID = sSTableReader.getSSTableMetadata().pendingRepair;
            if (!groups.containsKey(sessionID)) {
                groups.put(sessionID, Pair.create(new HashSet(), new HashSet()));
            }
            ((Set)((Pair)groups.get((Object)sessionID)).left).add(sSTableReader);
        }
        for (SSTableReader sSTableReader : added) {
            sessionID = sSTableReader.getSSTableMetadata().pendingRepair;
            if (!groups.containsKey(sessionID)) {
                groups.put(sessionID, Pair.create(new HashSet(), new HashSet()));
            }
            ((Set)((Pair)groups.get((Object)sessionID)).right).add(sSTableReader);
        }
        for (Map.Entry entry : groups.entrySet()) {
            AbstractCompactionStrategy strategy = this.getOrCreate((UUID)entry.getKey());
            Set groupRemoved = (Set)((Pair)entry.getValue()).left;
            Set groupAdded = (Set)((Pair)entry.getValue()).right;
            if (!groupRemoved.isEmpty()) {
                strategy.replaceSSTables(groupRemoved, groupAdded);
            } else {
                strategy.addSSTables(groupAdded);
            }
            this.removeSessionIfEmpty((UUID)entry.getKey());
        }
    }

    synchronized void startup() {
        this.strategies.values().forEach(AbstractCompactionStrategy::startup);
    }

    synchronized void shutdown() {
        this.strategies.values().forEach(AbstractCompactionStrategy::shutdown);
    }

    private int getEstimatedRemainingTasks(UUID sessionID, AbstractCompactionStrategy strategy) {
        if (this.canCleanup(sessionID)) {
            return 0;
        }
        return strategy.getEstimatedRemainingTasks();
    }

    int getEstimatedRemainingTasks() {
        int tasks = 0;
        for (Map.Entry entry : this.strategies.entrySet()) {
            tasks += this.getEstimatedRemainingTasks((UUID)entry.getKey(), (AbstractCompactionStrategy)entry.getValue());
        }
        return tasks;
    }

    int getMaxEstimatedRemainingTasks() {
        int tasks = 0;
        for (Map.Entry entry : this.strategies.entrySet()) {
            tasks = Math.max(tasks, this.getEstimatedRemainingTasks((UUID)entry.getKey(), (AbstractCompactionStrategy)entry.getValue()));
        }
        return tasks;
    }

    private RepairFinishedCompactionTask getRepairFinishedCompactionTask(UUID sessionID) {
        Preconditions.checkState(this.canCleanup(sessionID));
        AbstractCompactionStrategy compactionStrategy = this.get(sessionID);
        if (compactionStrategy == null) {
            return null;
        }
        Set<SSTableReader> sstables = compactionStrategy.getSSTables();
        long repairedAt = ActiveRepairService.instance.consistent.local.getFinalSessionRepairedAt(sessionID);
        LifecycleTransaction txn = this.cfs.getTracker().tryModify(sstables, OperationType.COMPACTION);
        return txn == null ? null : new RepairFinishedCompactionTask(this.cfs, txn, sessionID, repairedAt);
    }

    public CleanupTask releaseSessionData(Collection<UUID> sessionIDs) {
        ArrayList<Pair<UUID, RepairFinishedCompactionTask>> tasks = new ArrayList<Pair<UUID, RepairFinishedCompactionTask>>(sessionIDs.size());
        for (UUID session : sessionIDs) {
            if (!this.hasDataForSession(session)) continue;
            tasks.add(Pair.create(session, this.getRepairFinishedCompactionTask(session)));
        }
        return new CleanupTask(this.cfs, tasks);
    }

    synchronized int getNumPendingRepairFinishedTasks() {
        int count = 0;
        for (UUID sessionID : this.strategies.keySet()) {
            if (!this.canCleanup(sessionID)) continue;
            ++count;
        }
        return count;
    }

    synchronized AbstractCompactionTask getNextRepairFinishedTask() {
        for (UUID sessionID : this.strategies.keySet()) {
            if (!this.canCleanup(sessionID)) continue;
            return this.getRepairFinishedCompactionTask(sessionID);
        }
        return null;
    }

    synchronized AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        if (this.strategies.isEmpty()) {
            return null;
        }
        HashMap numTasks = new HashMap(this.strategies.size());
        ArrayList sessions = new ArrayList(this.strategies.size());
        for (Map.Entry entry : this.strategies.entrySet()) {
            if (this.canCleanup((UUID)entry.getKey())) continue;
            numTasks.put(entry.getKey(), this.getEstimatedRemainingTasks((UUID)entry.getKey(), (AbstractCompactionStrategy)entry.getValue()));
            sessions.add(entry.getKey());
        }
        if (sessions.isEmpty()) {
            return null;
        }
        sessions.sort((o1, o2) -> (Integer)numTasks.get(o2) - (Integer)numTasks.get(o1));
        UUID sessionID = (UUID)sessions.get(0);
        return this.get(sessionID).getNextBackgroundTask(gcBefore);
    }

    synchronized Collection<AbstractCompactionTask> getMaximalTasks(int gcBefore, boolean splitOutput) {
        if (this.strategies.isEmpty()) {
            return null;
        }
        ArrayList<AbstractCompactionTask> maximalTasks = new ArrayList<AbstractCompactionTask>(this.strategies.size());
        for (Map.Entry entry : this.strategies.entrySet()) {
            if (this.canCleanup((UUID)entry.getKey())) {
                maximalTasks.add(this.getRepairFinishedCompactionTask((UUID)entry.getKey()));
                continue;
            }
            Collection<AbstractCompactionTask> tasks = ((AbstractCompactionStrategy)entry.getValue()).getMaximalTask(gcBefore, splitOutput);
            if (tasks == null) continue;
            maximalTasks.addAll(tasks);
        }
        return !maximalTasks.isEmpty() ? maximalTasks : null;
    }

    Collection<AbstractCompactionStrategy> getStrategies() {
        return this.strategies.values();
    }

    Set<UUID> getSessions() {
        return this.strategies.keySet();
    }

    boolean canCleanup(UUID sessionID) {
        return !ActiveRepairService.instance.consistent.local.isSessionInProgress(sessionID);
    }

    synchronized Set<ISSTableScanner> getScanners(Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
        if (sstables.isEmpty()) {
            return Collections.emptySet();
        }
        HashMap<UUID, Set> sessionSSTables = new HashMap<UUID, Set>();
        for (SSTableReader sSTableReader : sstables) {
            UUID uUID = sSTableReader.getSSTableMetadata().pendingRepair;
            PendingRepairManager.checkPendingID(uUID);
            sessionSSTables.computeIfAbsent(uUID, k -> new HashSet()).add(sSTableReader);
        }
        HashSet<ISSTableScanner> scanners = new HashSet<ISSTableScanner>(sessionSSTables.size());
        try {
            for (Map.Entry entry : sessionSSTables.entrySet()) {
                scanners.addAll(this.getOrCreate((UUID)((UUID)entry.getKey())).getScanners((Collection<SSTableReader>)((Collection)entry.getValue()), ranges).scanners);
            }
        }
        catch (Throwable throwable) {
            ISSTableScanner.closeAllAndPropagate(scanners, throwable);
        }
        return scanners;
    }

    public boolean hasStrategy(AbstractCompactionStrategy strategy) {
        return ((ImmutableCollection)this.strategies.values()).contains(strategy);
    }

    public synchronized boolean hasDataForSession(UUID sessionID) {
        return this.strategies.containsKey(sessionID);
    }

    boolean containsSSTable(SSTableReader sstable) {
        if (!sstable.isPendingRepair()) {
            return false;
        }
        AbstractCompactionStrategy strategy = this.strategies.get(sstable.getPendingRepair());
        return strategy != null && strategy.getSSTables().contains(sstable);
    }

    public Collection<AbstractCompactionTask> createUserDefinedTasks(Collection<SSTableReader> sstables, int gcBefore) {
        Map<UUID, List<SSTableReader>> group = sstables.stream().collect(Collectors.groupingBy(s2 -> s2.getSSTableMetadata().pendingRepair));
        return group.entrySet().stream().map(g2 -> this.strategies.get(g2.getKey()).getUserDefinedTask((Collection)g2.getValue(), gcBefore)).collect(Collectors.toList());
    }

    @VisibleForTesting
    public synchronized boolean hasPendingRepairSSTable(UUID sessionID, SSTableReader sstable) {
        AbstractCompactionStrategy strat = this.strategies.get(sessionID);
        if (strat == null) {
            return false;
        }
        return strat.getSSTables().contains(sstable);
    }

    class RepairFinishedCompactionTask
    extends AbstractCompactionTask {
        private final UUID sessionID;
        private final long repairedAt;

        RepairFinishedCompactionTask(ColumnFamilyStore cfs, LifecycleTransaction transaction, UUID sessionID, long repairedAt) {
            super(cfs, transaction);
            this.sessionID = sessionID;
            this.repairedAt = repairedAt;
        }

        @VisibleForTesting
        UUID getSessionID() {
            return this.sessionID;
        }

        @Override
        protected void runMayThrow() throws Exception {
            boolean completed = false;
            boolean obsoleteSSTables = PendingRepairManager.this.isTransient && this.repairedAt > 0L;
            try {
                if (obsoleteSSTables) {
                    logger.info("Obsoleting transient repaired sstables for {}", (Object)this.sessionID);
                    Preconditions.checkState(Iterables.all(this.transaction.originals(), SSTableReader::isTransient));
                    this.transaction.obsoleteOriginals();
                } else {
                    logger.info("Moving {} from pending to repaired with repaired at = {} and session id = {}", new Object[]{this.transaction.originals(), this.repairedAt, this.sessionID});
                    this.cfs.getCompactionStrategyManager().mutateRepaired(this.transaction.originals(), this.repairedAt, ActiveRepairService.NO_PENDING_REPAIR, false);
                }
                completed = true;
            }
            finally {
                if (obsoleteSSTables) {
                    this.transaction.finish();
                } else {
                    this.transaction.abort();
                }
                if (completed) {
                    PendingRepairManager.this.removeSessionIfEmpty(this.sessionID);
                }
            }
        }

        @Override
        public CompactionAwareWriter getCompactionAwareWriter(ColumnFamilyStore cfs, Directories directories, LifecycleTransaction txn, Set<SSTableReader> nonExpiredSSTables) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected int executeInternal(ActiveCompactionsTracker activeCompactions) {
            this.run();
            return this.transaction.originals().size();
        }
    }

    public static class CleanupTask {
        private final ColumnFamilyStore cfs;
        private final List<Pair<UUID, RepairFinishedCompactionTask>> tasks;

        public CleanupTask(ColumnFamilyStore cfs, List<Pair<UUID, RepairFinishedCompactionTask>> tasks) {
            this.cfs = cfs;
            this.tasks = tasks;
        }

        public CleanupSummary cleanup() {
            HashSet<UUID> successful = new HashSet<UUID>();
            HashSet<UUID> unsuccessful = new HashSet<UUID>();
            for (Pair<UUID, RepairFinishedCompactionTask> pair : this.tasks) {
                UUID session = (UUID)pair.left;
                RepairFinishedCompactionTask task = (RepairFinishedCompactionTask)pair.right;
                if (task != null) {
                    try {
                        task.run();
                        successful.add(session);
                    }
                    catch (Throwable t) {
                        t = task.transaction.abort(t);
                        logger.error("Failed cleaning up " + session, t);
                        unsuccessful.add(session);
                    }
                    continue;
                }
                unsuccessful.add(session);
            }
            return new CleanupSummary(this.cfs, successful, unsuccessful);
        }

        public Throwable abort(Throwable accumulate) {
            for (Pair<UUID, RepairFinishedCompactionTask> pair : this.tasks) {
                accumulate = ((RepairFinishedCompactionTask)pair.right).transaction.abort(accumulate);
            }
            return accumulate;
        }
    }

    public static class IllegalSSTableArgumentException
    extends IllegalArgumentException {
        public IllegalSSTableArgumentException(String s2) {
            super(s2);
        }
    }
}

