/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Ints;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Supplier;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.io.compress.BufferType;
import org.apache.cassandra.io.compress.CompressionMetadata;
import org.apache.cassandra.io.compress.CorruptBlockException;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.util.AbstractReaderFileProxy;
import org.apache.cassandra.io.util.BufferManagingRebufferer;
import org.apache.cassandra.io.util.ChannelProxy;
import org.apache.cassandra.io.util.ChunkReader;
import org.apache.cassandra.io.util.MmappedRegions;
import org.apache.cassandra.io.util.Rebufferer;
import org.apache.cassandra.io.util.ThreadLocalByteBufferHolder;
import org.apache.cassandra.io.util.ThreadLocalReadAheadBuffer;
import org.apache.cassandra.utils.ChecksumType;
import org.apache.cassandra.utils.Closeable;

public abstract class CompressedChunkReader
extends AbstractReaderFileProxy
implements ChunkReader {
    final CompressionMetadata metadata;
    final int maxCompressedLength;
    final Supplier<Double> crcCheckChanceSupplier;

    protected CompressedChunkReader(ChannelProxy channel, CompressionMetadata metadata, Supplier<Double> crcCheckChanceSupplier) {
        super(channel, metadata.dataLength);
        this.metadata = metadata;
        this.maxCompressedLength = metadata.maxCompressedLength();
        this.crcCheckChanceSupplier = crcCheckChanceSupplier;
        assert (Integer.bitCount(metadata.chunkLength()) == 1);
    }

    protected CompressedChunkReader forScan() {
        return this;
    }

    @Override
    @VisibleForTesting
    public double getCrcCheckChance() {
        return this.crcCheckChanceSupplier.get();
    }

    boolean shouldCheckCrc() {
        double checkChance = this.getCrcCheckChance();
        return checkChance >= 1.0 || checkChance > 0.0 && checkChance > ThreadLocalRandom.current().nextDouble();
    }

    @Override
    public String toString() {
        return String.format("CompressedChunkReader.%s(%s - %s, chunk length %d, data length %d)", this.getClass().getSimpleName(), this.channel.filePath(), this.metadata.compressor().getClass().getSimpleName(), this.metadata.chunkLength(), this.metadata.dataLength);
    }

    @Override
    public int chunkSize() {
        return this.metadata.chunkLength();
    }

    @Override
    public BufferType preferredBufferType() {
        return this.metadata.compressor().preferredBufferType();
    }

    @Override
    public Rebufferer instantiateRebufferer(boolean isScan) {
        return new BufferManagingRebufferer.Aligned(isScan ? this.forScan() : this);
    }

    public static class Mmap
    extends CompressedChunkReader {
        protected final MmappedRegions regions;

        public Mmap(ChannelProxy channel, CompressionMetadata metadata, MmappedRegions regions, Supplier<Double> crcCheckChanceSupplier) {
            super(channel, metadata, crcCheckChanceSupplier);
            this.regions = regions;
        }

        @Override
        public void readChunk(long position, ByteBuffer uncompressed) {
            try {
                assert ((position & (long)(-uncompressed.capacity())) == position);
                assert (position <= this.fileLength);
                CompressionMetadata.Chunk chunk = this.metadata.chunkFor(position);
                MmappedRegions.Region region = this.regions.floor(chunk.offset);
                long segmentOffset = region.offset();
                int chunkOffset = Ints.checkedCast((long)(chunk.offset - segmentOffset));
                ByteBuffer compressedChunk = region.buffer();
                compressedChunk.position(chunkOffset).limit(chunkOffset + chunk.length);
                uncompressed.clear();
                try {
                    if (this.shouldCheckCrc()) {
                        int checksum = (int)ChecksumType.CRC32.of(compressedChunk);
                        compressedChunk.limit(compressedChunk.capacity());
                        if (compressedChunk.getInt() != checksum) {
                            throw new CorruptBlockException(this.channel.filePath(), chunk);
                        }
                        compressedChunk.position(chunkOffset).limit(chunkOffset + chunk.length);
                    }
                    if (chunk.length < this.maxCompressedLength) {
                        this.metadata.compressor().uncompress(compressedChunk, uncompressed);
                    } else {
                        uncompressed.put(compressedChunk);
                    }
                }
                catch (IOException e) {
                    throw new CorruptBlockException(this.channel.filePath(), chunk, e);
                }
                uncompressed.flip();
            }
            catch (CorruptBlockException e) {
                uncompressed.position(0).limit(0);
                throw new CorruptSSTableException((Throwable)e, this.channel.filePath());
            }
        }

        @Override
        public void close() {
            this.regions.closeQuietly();
            super.close();
        }
    }

    public static class Standard
    extends CompressedChunkReader {
        private final CompressedReader reader;
        private final CompressedReader scanReader;

        public Standard(ChannelProxy channel, CompressionMetadata metadata, Supplier<Double> crcCheckChanceSupplier) {
            super(channel, metadata, crcCheckChanceSupplier);
            this.reader = new RandomAccessCompressedReader(channel, metadata);
            int readAheadBufferSize = DatabaseDescriptor.getCompressedReadAheadBufferSize();
            this.scanReader = readAheadBufferSize > 0 && readAheadBufferSize > metadata.chunkLength() ? new ScanCompressedReader(channel, metadata, readAheadBufferSize) : null;
        }

        @Override
        protected CompressedChunkReader forScan() {
            if (this.scanReader != null) {
                this.scanReader.allocateResources();
            }
            return this;
        }

        @Override
        public void releaseUnderlyingResources() {
            if (this.scanReader != null) {
                this.scanReader.deallocateResources();
            }
        }

        @Override
        public void readChunk(long position, ByteBuffer uncompressed) {
            try {
                CompressedReader readFrom;
                assert ((position & (long)(-uncompressed.capacity())) == position);
                assert (position <= this.fileLength);
                CompressionMetadata.Chunk chunk = this.metadata.chunkFor(position);
                boolean shouldCheckCrc = this.shouldCheckCrc();
                CompressedReader compressedReader = readFrom = this.scanReader != null && this.scanReader.allocated() ? this.scanReader : this.reader;
                if (chunk.length < this.maxCompressedLength) {
                    ByteBuffer compressed = readFrom.read(chunk, shouldCheckCrc);
                    uncompressed.clear();
                    try {
                        this.metadata.compressor().uncompress(compressed, uncompressed);
                    }
                    catch (IOException e) {
                        throw new CorruptBlockException(this.channel.filePath(), chunk, e);
                    }
                } else {
                    uncompressed.position(0).limit(chunk.length);
                    if (this.channel.read(uncompressed, chunk.offset) != chunk.length) {
                        throw new CorruptBlockException(this.channel.filePath(), chunk);
                    }
                    if (shouldCheckCrc) {
                        uncompressed.flip();
                        int checksum = (int)ChecksumType.CRC32.of(uncompressed);
                        ByteBuffer scratch = ByteBuffer.allocate(4);
                        if (this.channel.read(scratch, chunk.offset + (long)chunk.length) != 4 || scratch.getInt(0) != checksum) {
                            throw new CorruptBlockException(this.channel.filePath(), chunk);
                        }
                    }
                }
                uncompressed.flip();
            }
            catch (CorruptBlockException e) {
                uncompressed.position(0).limit(0);
                throw new CorruptSSTableException((Throwable)e, this.channel.filePath());
            }
        }

        @Override
        public void close() {
            this.reader.close();
            if (this.scanReader != null) {
                this.scanReader.close();
            }
            super.close();
        }
    }

    private static class ScanCompressedReader
    implements CompressedReader {
        private final ChannelProxy channel;
        private final ThreadLocalByteBufferHolder bufferHolder;
        private final ThreadLocalReadAheadBuffer readAheadBuffer;

        private ScanCompressedReader(ChannelProxy channel, CompressionMetadata metadata, int readAheadBufferSize) {
            this.channel = channel;
            this.bufferHolder = new ThreadLocalByteBufferHolder(metadata.compressor().preferredBufferType());
            this.readAheadBuffer = new ThreadLocalReadAheadBuffer(channel, readAheadBufferSize, metadata.compressor().preferredBufferType());
        }

        @Override
        public ByteBuffer read(CompressionMetadata.Chunk chunk, boolean shouldCheckCrc) throws CorruptBlockException {
            int length = shouldCheckCrc ? chunk.length + 4 : chunk.length;
            ByteBuffer compressed = this.bufferHolder.getBuffer(length);
            int copied = 0;
            while (copied < length) {
                this.readAheadBuffer.fill(chunk.offset + (long)copied);
                int leftToRead = length - copied;
                if (this.readAheadBuffer.remaining() >= leftToRead) {
                    copied += this.readAheadBuffer.read(compressed, leftToRead);
                    continue;
                }
                copied += this.readAheadBuffer.read(compressed, this.readAheadBuffer.remaining());
            }
            compressed.flip();
            compressed.limit(chunk.length);
            if (shouldCheckCrc) {
                int checksum = (int)ChecksumType.CRC32.of(compressed);
                compressed.limit(length);
                if (compressed.getInt() != checksum) {
                    throw new CorruptBlockException(this.channel.filePath(), chunk);
                }
                compressed.position(0).limit(chunk.length);
            }
            return compressed;
        }

        @Override
        public void allocateResources() {
            this.readAheadBuffer.allocateBuffer();
        }

        @Override
        public void deallocateResources() {
            this.readAheadBuffer.clear(true);
        }

        @Override
        public boolean allocated() {
            return this.readAheadBuffer.hasBuffer();
        }

        @Override
        public void close() {
            this.readAheadBuffer.close();
        }
    }

    private static class RandomAccessCompressedReader
    implements CompressedReader {
        private final ChannelProxy channel;
        private final ThreadLocalByteBufferHolder bufferHolder;

        private RandomAccessCompressedReader(ChannelProxy channel, CompressionMetadata metadata) {
            this.channel = channel;
            this.bufferHolder = new ThreadLocalByteBufferHolder(metadata.compressor().preferredBufferType());
        }

        @Override
        public ByteBuffer read(CompressionMetadata.Chunk chunk, boolean shouldCheckCrc) throws CorruptBlockException {
            int length = shouldCheckCrc ? chunk.length + 4 : chunk.length;
            ByteBuffer compressed = this.bufferHolder.getBuffer(length);
            if (this.channel.read(compressed, chunk.offset) != length) {
                throw new CorruptBlockException(this.channel.filePath(), chunk);
            }
            compressed.flip();
            compressed.limit(chunk.length);
            if (shouldCheckCrc) {
                int checksum = (int)ChecksumType.CRC32.of(compressed);
                compressed.limit(length);
                if (compressed.getInt() != checksum) {
                    throw new CorruptBlockException(this.channel.filePath(), chunk);
                }
                compressed.position(0).limit(chunk.length);
            }
            return compressed;
        }
    }

    protected static interface CompressedReader
    extends Closeable {
        default public void allocateResources() {
        }

        default public void deallocateResources() {
        }

        default public boolean allocated() {
            return false;
        }

        @Override
        default public void close() {
        }

        public ByteBuffer read(CompressionMetadata.Chunk var1, boolean var2) throws CorruptBlockException;
    }
}

